#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "libfma.h"
#include "lf_fabric.h"
#include "lf_product_def.h"

#include "linktest.h"
#include "gmstuff.h"

extern char *Test_x1;
extern int Test_p1;
extern char *Test_x2;
extern int Test_p2;
extern int Hport;

int Ntests;
int Nretries;
extern int Show_route;
extern char *One_switch;

int test_this_route(int ifc, unsigned char *route, int rlen, int report);
int do_this_xbar(int ifc, struct lf_xbar *xp, int in_port, unsigned char *route,
	         int rlen, int depth);

static void
print_route(
  unsigned char *route,
  int rlen)
{
  int i;

  printf("route:");
  for (i=0; i<rlen; ++i) {
    printf(" %d", LF_ROUTE_TO_DELTA(route[i]));
  }
  printf("\n");
}

static void
print_result(
  struct lf_xbar *xp,
  int port,
  int result)
{
  struct lf_xbar *oxp;
  int oport;
  struct lf_linecard *lp;
  struct lf_linecard *olp;

  oxp = LF_XBAR(xp->topo_ports[port]);
  oport = xp->topo_rports[port];

  lp = xp->linecard;
  olp = oxp->linecard;


  /* If phys connection is same xbar, then its an internal link */
  if (xp->phys_ports[port] == LF_NODE(oxp)) {

    printf("The internal link on enclosure %s between\n",
	lp->enclosure->name);
    printf("\tslot %d, xbar %d, port %d    and\n",
	lf_slot_display_no(lp), xp->xbar_no, port);
    printf("\tslot %d, xbar %d, port %d\n",
	lf_slot_display_no(olp), oxp->xbar_no, oport);

  } else {
    struct lf_xcvr *xcp;
    struct lf_xcvr *oxcp;
    int xcport;

    xcp = LF_XCVR(xp->phys_ports[port]);
    oxcp = LF_XCVR(oxp->phys_ports[oport]);

    /* convert internal port number to external */
    xcport = xp->phys_rports[port] - xcp->num_conns;

    /* sanity checks */
    if (xcp->ln_type != LF_NODE_LC_XCVR) {
      printf("%s,s%d,x%d,p%d not connected to xcvr??\n", lp->enclosure->name,
	lf_slot_display_no(lp), xp->xbar_no, port);
      exit(1);
    }
    if (oxcp->ln_type != LF_NODE_LC_XCVR) {
      printf("%s,s%d,x%d,p%d not connected to xcvr??\n", olp->enclosure->name,
	lf_slot_display_no(olp), oxp->xbar_no, oport);
      exit(1);
    }
    if (xcp->ports[xcport] != LF_NODE(oxcp)) {
      printf("xcvrs not connected as expected??\n");
      printf("btwn %s,s%d,x%d,p%d", lp->enclosure->name,
	lf_slot_display_no(lp), xp->xbar_no, port);
      printf("and %s,s%d,x%d,p%d\n", olp->enclosure->name,
	lf_slot_display_no(olp), oxp->xbar_no, oport);
      exit(1);
    }

    /* change the linecard pointers to be for the xcvrs */
    lp = xcp->p.linecard;
    olp = oxcp->p.linecard;

    printf("The %s link between\n", lp->def->link_type);
    printf("\t%s, slot %d, port %d:%d   and\n",
	lp->enclosure->name, lf_slot_display_no(lp),
	lp->xcvr_labels[xcp->port],
	xp->phys_rports[port] - xcp->num_conns);
    printf("\t%s, slot %d, port %d:%d\n",
	olp->enclosure->name, lf_slot_display_no(olp),
	olp->xcvr_labels[oxcp->port],
	oxp->phys_rports[oport] - oxcp->num_conns);

  }

  /* Now say what's wrong with the link */
  if (result == LT_RES_NO_LINK) {
    printf("is currently passing no traffic.\n");
  } else if (result == LT_RES_MARGINAL_LINK) {
    printf("is exhibiting high packet loss\n");
  } else if (result == LT_RES_SLOW_LINK) {
    printf("is passing traffic slowly.\n");
  }

  printf("\n");
}

static void inline
alloc_xbar_user(
  struct lf_xbar *xp)
{
  struct lt_xbar *up;

  if (xp->user != NULL) return;

  LF_CALLOC(up, struct lt_xbar, 1);
  LF_CALLOC(up->bad, int, xp->num_ports);
  LF_CALLOC(up->tested, int, xp->num_ports);

  xp->user = up;
  return;

 except:
  exit(1);
}

int
do_this_xbar(
  int ifc,
  struct lf_xbar *xp,
  int in_port,
  unsigned char *route,
  int rlen,
  int depth)
{
  int total_rlen;
  int i;
  struct lf_xbar *newxp;
  int new_port;
  int count;
  int result;

  /* printf("doing xbar %s, in_port: %d, depth: %d\n", xp->name, in_port, depth); */

  alloc_xbar_user(xp);

  /* if depth is zero, this is a leaf, execute the tests */
  if (depth == 0) {
    
    /* U-turn byte. If using xbar32 ID packets, use 0xA0, for raw data packets,
     * use a 0 to redirect the packet back to where it came from
     */
    route[rlen] = (xp->num_ports > 16) ? 0xA0 : DELTA_TO_ROUTE(0);
route[rlen] = 0x80;

    /* then build path back to origin */
    for (i=0; i<rlen; ++i) {
      route[rlen+i+1] = DELTA_TO_ROUTE(-ROUTE_TO_DELTA(route[rlen-i-1]));
    }
    total_rlen = 2 * rlen + 1;

#if 0
    if ((Test_x1 == NULL) ||
	(Test_p1 == in_port && Test_p2 == xp->topo_rports[in_port] &&
	 strcmp(Test_x1, xp->name) == 0 &&
	 strcmp(Test_x2, xp->port[in_port].ptr.x->name) == 0) ||
	(Test_p2 == in_port && Test_p1 == xp->port[in_port].conn_port &&
	 strcmp(Test_x2, xp->name) == 0 &&
	 strcmp(Test_x1, xp->port[in_port].ptr.x->name) == 0)) {
    }
#else
    if (1) {
#endif

      /* perform the test, report if fails */
      ++Ntests;	/* keep track of links tested */

      if (One_switch == NULL
          || strcmp(xp->linecard->enclosure->name, One_switch) == 0) {

	result = test_this_route(ifc, route, total_rlen, 1);
	if (result != LT_RES_OK) {

	  /* mark both ends bad */
	  LT_XBAR(xp->user)->bad[in_port] = TRUE;
	  if (xp->topo_ports[in_port]->ln_type == LF_NODE_XBAR) {
	    struct lf_xbar *oxp;

	    oxp = LF_XBAR(xp->topo_ports[in_port]);
	    alloc_xbar_user(oxp);
	    LT_XBAR(oxp->user)->bad[xp->topo_rports[in_port]] = TRUE;
	  }

	  if (Show_route) print_route(route, total_rlen);
	  print_result(xp, in_port, result);
	}
      }
    }

    return 1;



  /* non-zero depth means to branch out to next level of non-tested xbars */
  } else {

    count = 0;		/* count how many new ports we tested */

    /* Now, work on each attached xbar */
    for (i=0; i<xp->num_ports; ++i) {

      /* skip non-xbars and bad ports */
      if (xp->topo_ports[i] == NULL
	  || xp->topo_ports[i]->ln_type != LF_NODE_XBAR
	  || LT_XBAR(xp->user)->bad[i]) {
	continue;
      }

      /* xbar and port we will process next */
      newxp = LF_XBAR(xp->topo_ports[i]);
      new_port = xp->topo_rports[i];

      /* route to the new xbar */
      route[rlen] = DELTA_TO_ROUTE(i - in_port);

      /* depth of 1 means the links will be tested */
      if (depth == 1) {

	/* If this port already tested, continue */
	if (LT_XBAR(xp->user)->tested[i]) {
	  continue;
	}
	alloc_xbar_user(newxp);

	LT_XBAR(xp->user)->tested[i] = TRUE;	/* this link has been tested */
	LT_XBAR(newxp->user)->tested[new_port] = TRUE;
      }

      /* process the xbar */
      count += do_this_xbar(ifc, newxp, new_port, route, rlen+1, depth-1);
    }
  }

  /* return the number of tests performed */
  return count;
}

void
check_all_links(
  struct lf_fabric *fp,
  struct lf_nic *nicp,
  int ifc)
{
  struct lf_xbar *xb;
  unsigned char route[32];
  int port;
  int depth;
  int result;
  int rc;

  /* This is the host we start from */
  xb = LF_XBAR(nicp->topo_ports[ifc]);
  port = nicp->topo_rports[ifc];

  /* make sure we have connection */
  route[0] = 0x80;
  result = test_this_route(ifc, route, 1, 1);
  if (result != LT_RES_OK) {
    printf("This host seems disconnected.\n");
    return;
  }

  /* keep running the checker at increasing depths until nothing found */
  Ntests = 0;
  Nretries = 0;
  depth = 0;
  do {
    rc = do_this_xbar(ifc, xb, port, route, 0, depth);
    ++depth;
  } while (rc > 0);

  printf ("%d links checked, %d retries.\n", Ntests, Nretries);
}
